<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TimeLimitedOffer;
use App\Models\OfferClaim;
use Illuminate\Http\Request;

class TimeOfferController extends Controller
{
    // Show list of all offers
    public function index()
    {
        $pageTitle = 'Time Limited Offers';
        $offers = TimeLimitedOffer::orderBy('created_at', 'desc')->paginate(getPaginate());
        
        return view('admin.time_offers.index', compact('pageTitle', 'offers'));
    }

    // Show create form
    public function create()
    {
        $pageTitle = 'Create New Offer';
        return view('admin.time_offers.create', compact('pageTitle'));
    }

    // Store new offer
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'bonus_amount' => 'required|numeric|min:0',
            'required_invites' => 'required|integer|min:1',
            'start_time' => 'required', // Datepicker sends string
            'end_time' => 'required',
            'status' => 'required|in:0,1'
        ]);

        TimeLimitedOffer::create([
            'title' => $request->title,
            'bonus_amount' => $request->bonus_amount,
            'required_invites' => $request->required_invites,
            'start_time' => $request->start_time, // Carbon/Eloquent handles casting
            'end_time' => $request->end_time,
            'status' => $request->status
        ]);

        $notify[] = ['success', 'Offer created successfully'];
        return redirect()->route('admin.time.offers.index')->withNotify($notify);
    }

    // Show edit form
    public function edit($id)
    {
        $pageTitle = 'Edit Offer';
        $offer = TimeLimitedOffer::findOrFail($id); // Standard find
        
        return view('admin.time_offers.edit', compact('pageTitle', 'offer'));
    }

    // Update offer
    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'bonus_amount' => 'required|numeric|min:0',
            'required_invites' => 'required|integer|min:1',
            'start_time' => 'required',
            'end_time' => 'required',
            'status' => 'required|in:0,1'
        ]);

        $offer = TimeLimitedOffer::findOrFail($id);
        $offer->update([
            'title' => $request->title,
            'bonus_amount' => $request->bonus_amount,
            'required_invites' => $request->required_invites,
            'start_time' => $request->start_time,
            'end_time' => $request->end_time,
            'status' => $request->status
        ]);

        $notify[] = ['success', 'Offer updated successfully'];
        return redirect()->route('admin.time.offers.index')->withNotify($notify);
    }

    // Delete offer
    public function destroy($id)
    {
        $offer = TimeLimitedOffer::findOrFail($id);
        $offer->delete();

        $notify[] = ['success', 'Offer deleted successfully'];
        return back()->withNotify($notify);
    }

    // View offer claims
    public function claims($id)
    {
        $pageTitle = 'Offer Claims';
        $offer = TimeLimitedOffer::findOrFail($id);
        
        $claims = OfferClaim::where('offer_id', $offer->id)
                          ->with('user')
                          ->orderBy('created_at', 'desc')
                          ->paginate(getPaginate());
        
        return view('admin.time_offers.claims', compact('pageTitle', 'offer', 'claims'));
    }

    // Toggle status
    public function toggleStatus($id)
    {
        $offer = TimeLimitedOffer::findOrFail($id);
        
        $offer->status = !$offer->status;
        $offer->save();

        $notify[] = ['success', 'Status updated successfully'];
        return back()->withNotify($notify);
    }
}
