<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\TimeLimitedOffer;
use App\Models\OfferClaim;
use App\Models\User;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TimeOfferController extends Controller
{
    // Show active offers to user
    public function index()
    {
        $pageTitle = 'Active Offers';
        $user = Auth::user();
        
        // Get active offers (Visible if Enabled by Admin)
        $activeOffers = TimeLimitedOffer::where('status', 1)->get();
        
        // Get user's claims
        $userClaims = OfferClaim::where('user_id', $user->id)
                               ->whereIn('offer_id', $activeOffers->pluck('id'))
                               ->get()
                               ->keyBy('offer_id');
        
        // Calculate user's paid invites during offer period
        $offerData = [];
        foreach ($activeOffers as $offer) {
            $claim = isset($userClaims[$offer->id]) ? $userClaims[$offer->id] : null;
            $paidInvitesCount = $claim ? $claim->invites_completed : 0;
            $claimed = $claim && $claim->bonus_claimed > 0;
            
            $offerData[] = [
                'offer' => $offer,
                'paid_invites' => $paidInvitesCount,
                'claimed' => $claimed,
                'can_claim' => !$claimed && $paidInvitesCount >= $offer->required_invites
            ];
        }
        
        return view('templates.ptc_diamond.user.time_offers.index', compact('pageTitle', 'offerData'));
    }

    // Claim bonus
    public function claim($id)
    {
        $general = gs();
        $user = Auth::user();
        $offer = TimeLimitedOffer::findOrFail($id);
        
        // Check if offer is active (Status based only, ignoring date validation for claim)
        if (!$offer->status) {
            $notify[] = ['error', 'This offer is currently disabled'];
            return back()->withNotify($notify);
        }
        
        // Find claim record
        $claim = OfferClaim::where('user_id', $user->id)
                          ->where('offer_id', $offer->id)
                          ->first();
        
        if ($claim && $claim->bonus_claimed > 0) {
            $notify[] = ['error', 'You have already claimed this offer'];
            return back()->withNotify($notify);
        }
        
        // Count paid invites
        $paidInvitesCount = $claim ? $claim->invites_completed : 0;
        
        // Check if user has completed required invites
        if ($paidInvitesCount < $offer->required_invites) {
            $notify[] = ['error', 'You need ' . $offer->required_invites . ' paid invites to claim this bonus'];
            return back()->withNotify($notify);
        }
        
        // Credit bonus to user
        $user->balance += $offer->bonus_amount;
        $user->save();
        
        // Create transaction record
        $transaction = new Transaction();
        $transaction->user_id = $user->id;
        $transaction->amount = $offer->bonus_amount;
        $transaction->post_balance = $user->balance;
        $transaction->charge = 0;
        $transaction->trx_type = '+';
        $transaction->details = 'Bonus from offer: ' . $offer->title;
        $transaction->trx = getTrx();
        $transaction->remark = 'time_offer_bonus';
        $transaction->save();
        
        // Update claim record
        if(!$claim) {
             // Should not happen if count > 0, but for safety
             $claim = new OfferClaim();
             $claim->user_id = $user->id;
             $claim->offer_id = $offer->id;
             $claim->invites_completed = $paidInvitesCount;
        }
        $claim->bonus_claimed = $offer->bonus_amount;
        $claim->claimed_at = now();
        $claim->save();
        
        $notify[] = ['success', 'Bonus claimed successfully! ' . showAmount($offer->bonus_amount) . ' ' . $general->cur_text . ' added to your account'];
        return back()->withNotify($notify);
    }
}
