<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class TimeLimitedOffer extends Model
{
    protected $table = 'time_limited_offers';
    
    protected $fillable = [
        'title',
        'bonus_amount',
        'required_invites',
        'start_time',
        'end_time',
        'status'
    ];

    protected $casts = [
        'bonus_amount' => 'decimal:8',
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'status' => 'boolean'
    ];

    // Check if offer is currently active
    public function isActive()
    {
        $now = now();
        return $this->status && 
               $this->start_time <= $now && 
               $this->end_time >= $now;
    }

    // Get remaining time in seconds
    public function getRemainingTimeAttribute()
    {
        if (!$this->isActive()) {
            return 0;
        }
        return max(0, $this->end_time->diffInSeconds(now()));
    }

    // Get claims for this offer
    public function claims()
    {
        return $this->hasMany(OfferClaim::class, 'offer_id');
    }

    // Scope for active offers
    public function scopeActive($query)
    {
        $now = now();
        return $query->where('status', 1)
                    ->where('start_time', '<=', $now)
                    ->where('end_time', '>=', $now);
    }
}
